////////////////////////////////////////////////
//
// ClusterMgrDBUtils.java
//	
package Alkindi.Services.ServicesImpl;

import java.util.*;
import java.io.*;
import java.sql.*;
import javax.transaction.*;
import Alkindi.Data.*;
import Alkindi.Services.*;
import Alkindi.Services.Util.*;
import Alkindi.Services.InternalData.*;
import oracle.jdbc.driver.OracleTypes;

/* 
$Header: ClusterMgrDBUtils.java, 44, 5/8/01 9:56:14 AM, Schwartz, Joe$
$Log: 
 44   Alkindi Development1.43        5/8/01 9:56:14 AM    Schwartz, Joe  
      Changed use of InternalEJBManager to InternalComponentManager. Removed
      dependencies on EJB from these classes. Cleaned up comments.
 43   Alkindi Development1.42        4/26/01 2:37:15 PM   Schwartz, Joe  
      Modifed to account for new InternalData package.
 42   Alkindi Development1.41        4/26/01 1:11:13 PM   Schwartz, Joe   Moved
      into new ServicesImpl package.
 41   Alkindi Development1.40        4/26/01 11:18:29 AM  Schwartz, Joe  
      Changed to account for connection being set to autocommit; removed several
      calls to con.commit().
 40   Alkindi Development1.39        4/25/01 4:42:39 PM   Schwartz, Joe   Added
      con.commit() calls to methods because they are no longer being committed
      as part of a transaction.
 39   Alkindi Development1.38        3/9/01 12:38:13 PM   Schwartz, Joe    
 38   Alkindi Development1.37        2/27/01 6:08:49 PM   Schwartz, Joe  
      Corrected comments.
 37   Alkindi Development1.36        2/22/01 5:23:46 PM   Schwartz, Joe  
      Removed autocommit from writeVMeansForPC. Removed some unused methods.
      Changed method getCoreEvalsForPC to use new SP getCoreProds.
 36   Alkindi Development1.35        2/22/01 3:15:17 PM   Schwartz, Joe   Added
      commit to the writeVMeansForPC method.
 35   Alkindi Development1.34        2/13/01 5:55:30 PM   Schwartz, Joe  
      Changed to account for new Product id int type and SparseRatingsArray.
 34   Alkindi Development1.33        2/12/01 10:38:22 AM  Schwartz, Joe  
      Changed to use new stored Procs.
 33   Alkindi Development1.32        2/6/01 11:53:48 AM   Schwartz, Joe   Adding
      new Product filling.
 32   Alkindi Development1.31        1/25/01 9:58:05 PM   Schwartz, Joe  
      Working on speed, sensibility.
 31   Alkindi Development1.30        1/22/01 1:46:47 PM   Schwartz, Joe  
      Modified to account for new RatingSpacePoint base class of VMeans &
      UserCluster.
 30   Alkindi Development1.29        1/4/01 2:50:00 PM    Schwartz, Joe  
      Removed transaction parameter from methods. Added connection commit()
      calls to commit changes from stored procs.
 29   Alkindi Development1.28        1/4/01 2:12:27 PM    Schwartz, Joe  
      Changed to pass LogManager instance to CDB constructor. Changed
      transaction handling in recalcStats.
 28   Alkindi Development1.27        12/28/00 1:46:41 PM  Schwartz, Joe  
      Changed to use javax.transaction.*
 27   Alkindi Development1.26        12/28/00 1:09:26 PM  Schwartz, Joe   Added
      Version Control header info.
 26   Alkindi Development1.25        12/28/00 12:05:51 PM Schwartz, Joe   
 25   Alkindi Development1.24        12/27/00 6:10:12 PM  Schwartz, Joe   
 24   Alkindi Development1.23        12/26/00 6:57:10 PM  Schwartz, Joe   
 23   Alkindi Development1.22        12/26/00 5:36:58 PM  Schwartz, Joe   
 22   Alkindi Development1.21        12/21/00 2:24:27 PM  Schwartz, Joe   Fixed
      fillFunction convergence. Changed cluster convergence test to use 
      unfilled VMeans. Added better XAction mgt.
 21   Alkindi Development1.20        12/20/00 5:44:53 PM  Schwartz, Joe   
 20   Alkindi Development1.19        12/18/00 12:05:34 PM Schwartz, Joe   Moved
      internal data classes to Utils package & regenerated classes from Rose.
 19   Alkindi Development1.18        12/18/00 11:22:40 AM Schwartz, Joe   
 18   Alkindi Development1.17        12/18/00 11:22:08 AM Schwartz, Joe   
 17   Alkindi Development1.16        12/3/00 5:27:55 PM   Schwartz, Joe   
 16   Alkindi Development1.15        12/2/00 9:21:49 PM   Schwartz, Joe   
 15   Alkindi Development1.14        12/2/00 8:31:41 PM   Schwartz, Joe   
 14   Alkindi Development1.13        12/2/00 7:37:25 PM   Schwartz, Joe   
 13   Alkindi Development1.12        12/2/00 1:02:26 PM   Schwartz, Joe   
 12   Alkindi Development1.11        12/2/00 10:47:12 AM  Schwartz, Joe   
 11   Alkindi Development1.10        11/29/00 2:06:34 PM  Schwartz, Joe  
      Changed to use stored procs for insert/update operations.
 10   Alkindi Development1.9         11/20/00 10:52:34 PM Schwartz, Joe   
 9    Alkindi Development1.8         11/14/00 12:53:12 PM Schwartz, Joe   
 8    Alkindi Development1.7         11/6/00 7:05:58 PM   Schwartz, Joe   
 7    Alkindi Development1.6         11/3/00 1:29:40 PM   Schwartz, Joe   Got
      rid of dependence on Parse2 and Parse3 objects. Added more error handling.
 6    Alkindi Development1.5         11/1/00 11:09:08 PM  Schwartz, Joe   
 5    Alkindi Development1.4         10/22/00 10:38:39 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 4    Alkindi Development1.3         10/22/00 10:34:08 AM Schwartz, Joe   
 3    Alkindi Development1.2         10/17/00 2:22:47 PM  Schwartz, Joe  
      Delivery 10.16.00
 2    Alkindi Development1.1         10/7/00 4:34:29 PM   Schwartz, Joe   
 1    Alkindi Development1.0         10/7/00 4:29:09 PM   Schwartz, Joe   
$
$NoKeywords$
 */

/**
 * A collection of database utilities related to the ClusterManager.
 */
public class ClusterMgrDBUtils 
{
	private LogManager logger = null;
	
	/**
	 * Number of products to use in Alkindex calculations (an Engine property).
	 */
	private static int propAlkindexTopNProds;
	private final String cName = "ClusterMgrDBUtils";
	
	/**
	 * Constructor to read Engine properties.
	 * @param logMgr The LogManager instance to use in logging.
	 * @roseuid 3A4B660B038A
	 */
	public ClusterMgrDBUtils(final Alkindi.Services.Util.LogManager logMgr) throws AlkExcept 
	{
		propAlkindexTopNProds = PropertyManager.getInt("alkindex.NUM_TOP_PRODS");
		logger = logMgr;
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.sp_INS_Cluster_User</b>.
	 * 	@param user The SystemUser to cluster.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C01D4
	 */
	public void clusterSingleUser(SystemUser user) throws AlkExcept 
	{
		final String mName = "clusterSingleUserDB";
		Connection con = null;
		try {
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_CLUSTER.sp_INS_Cluster_User(?)}");

			cs.setLong(1, user.id);
			cs.execute();
			cs.close();
       	} 
		
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("Exeption in " + mName + ": " + e.getMessage(), 2105);
		}
		finally {
			try { con.close(); } catch (Exception e2) {}
		}
	}
	
	/**
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660D003E
	 */
	public void delRelUserCluster() throws AlkExcept 
	{
		final String mName = "delRelUserCluster";
		Connection con = null;
		try {
			
			con = AlkConn.getConnection();
			PreparedStatement ps = con.prepareStatement("delete from REL_USER_CLUSTER");
			ps.executeUpdate();
			ps.close();
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2119);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.getCoreProdEvals</b>.
	 * 	@param pc The ProductCluster in question.
	 * 	@return UserRatingsList
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660B038B
	 */
	public UserRatingsList getCoreEvalsForPC(ProductCluster pc) throws AlkExcept 
	{
		final String mName = "getCoreEvalsForPC";

      	UserRatingsList url = new UserRatingsList();
		Connection con = null;

        try {
			con = AlkConn.getConnection();
			//	First read in ratings:
			//
			CallableStatement cs  = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.getCoreProdEvals(?,?,?)}");

			cs.setInt(1, pc.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
	
			ResultSet rs    = (ResultSet) cs.getObject(2);
       		int  error = cs.getInt(3);
			if (error != 0)
				throw new AlkExcept("Error in pkgALKINDI_CLUSTER.getCoreProdEvals.", error);
			
			//	Simply add each evaluation to the UserRatingsList object,
			//	which internally add them into the correct users' list.
			//
			int evalCount = 0;
			while (rs.next()){
				SystemUser su = new SystemUser(rs.getInt(AlkConn.COL_UID));
				Product prod = new Product(rs.getInt(AlkConn.COL_PID));
				Rating rat = new Rating(su, prod, rs.getInt(AlkConn.COL_EVALSCALEID));
				url.addRating(rat);
				evalCount ++;
			} 

			rs.close();
			cs.close();
			
			//	Now read in products for the product cluster
			//
			cs = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.getCoreProds(?,?, ?) }");
			cs.setInt(1, pc.id);
			cs.registerOutParameter(2, OracleTypes.CURSOR);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();

			rs    = (ResultSet) cs.getObject(2);
			while(rs.next()) {
				Product prod = new Product(rs.getInt(AlkConn.COL_PID));
				url.products.add(prod);
			}
			rs.close();
			cs.close();

       	} //END try
			
		catch (SQLException se) {
           	throw new AlkExcept(se.getMessage(), 6004);
		}
       	catch (NullPointerException ne) {
			throw new AlkExcept("Exception in " + mName + ": " + ne.getMessage(), 6017, ne);
        }
		finally {
			try { con.close(); } catch (Exception e) { }
		}
       	return url;
	}
	
	/**
	 * Returns the means of the Core Products for the given ProductsCluster. 
	 * @param pcID The ID of the ProductCluster in question.
	 * @return VMeansList
	 * @roseuid 3A7990EF0251
	 */
	public VMeansList getCoreVMeans(final int pcID) throws AlkExcept 
	{
		final String mName = "getCoreVMeans";
		//	Declare return value
		//
		VMeansList coreVMeans = null;
		Connection con = null;
		try {
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.getCoreProdUCStats(?,?,?)}");
			cs.setInt(1, pcID);
			cs.registerOutParameter(2, OracleTypes.CURSOR);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(3);
			if (err != 0) 
				throw new AlkExcept("Unexpected error in pkg_ALKINDI_CLUSTER.getCoreProdUCStats", 6006); 
			ResultSet rs = (ResultSet) cs.getObject(2);
			coreVMeans = parseVMLFromRS(rs);
			rs.close();
			cs.close();
		}
		catch (SQLException se) {
			throw new AlkExcept("SQLException in " + mName + ": " + se.getMessage(), 6004);
		}
		finally {
			try { con.close(); } catch (Exception e) {}
		}
		return coreVMeans;
	}
	
	/**
	 * 	Retrieves the number of UserClusters in the given ProductCluster.
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.sp_CNT_UC_BY_PC</b>
	 * 	@param pc The ProductCluster in question.
	 * 	@return int
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660B03D8
	 */
	public int getNumUCInPC(ProductCluster pc) throws AlkExcept 
	{
		final String mName = "getNumUCInPC";
     	int UCinPC;
		Connection con = null;
		try{
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.sp_CNT_UC_BY_PC (?,?,?)}");
			cs.setInt(1,pc.id);
			cs.registerOutParameter(2, Types.INTEGER);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			UCinPC = cs.getInt(2);
			int error = cs.getInt(3);
			if (error == -1) {
				throw new AlkExcept("Error in pkg_ALKINDI_CLUSTER.sp_CNT_UC_BY_PC", 7200);
			}
			cs.close();
			con.close();

		} 
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in ClusterMgrDBUtils.getNumUCInPC: " + se.getMessage(), 6004);
        }
		finally {
			try { con.close(); } catch (Exception e) {}
		}
		return UCinPC;
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.sp_CNT_User_By_UC_INDEX</b>.
	 * 	@param ucIdx The index of the UserCluster in question.
	 * 	@param pcID The ProductCluster in question.
	 * 	@return int
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C000F
	 */
	public int getNumUsersInUC(final int ucIdx, final int pcID) throws AlkExcept 
	{
		final String mName = "getNumUsersInUC";
		
		int o_numusers;
        int error;
		Connection con = null;
		try{
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.sp_CNT_User_By_UC_INDEX (?,?,?,?)}");
			cs.setInt(1,ucIdx);
			cs.setInt(2,pcID);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.registerOutParameter(4, Types.INTEGER);
			cs.execute();

			o_numusers = cs.getInt(3);
			error = cs.getInt(4);
			if (error == -1) {
				throw new AlkExcept("Error in sp_CNT_User_By_UC_INDEX", 7200);
			}
			cs.close();
		} 
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
        }
		finally {
			try { con.close(); } catch (Exception e) {}
		}
		return o_numusers;
	}
	
	/**
	 * Retrieves the means of the Recommendable Products for the given ProductsCluster. 
	 * @param pcID The ID of the ProductCluster in question.
	 * @return VMeansList
	 * @roseuid 3A7991280203
	 */
	public final VMeansList getRecVMeans(final int pcID) throws AlkExcept 
	{
		final String mName = "getRecVMeans";
		
		//	Declare return value
		//
		VMeansList recVMeans = null;
		Connection con = null;
		try {
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.getRecProdUCStats(?,?,?) }");
			cs.setLong(1, pcID);
			cs.registerOutParameter(2, OracleTypes.CURSOR);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(3);
			if (err != 0) 
				throw new AlkExcept("Unexpected error in pkg_ALKINDI_CLUSTER.getRecProdUCStats", 6006); 
			ResultSet rs = (ResultSet) cs.getObject(2);
			recVMeans = parseVMLFromRS(rs);
		}
		catch (SQLException se) {
			throw new AlkExcept("SQLException in " + mName + ": " + se.getMessage(), 6004);
		}
		finally {
			try { con.close(); } catch (Exception e) {}
		}
		return recVMeans;
	}
	
	/**
	 * 	Returns the unique UserCluster ID corresponding to the index relative to the given ProductCluster.
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.sp_SEL_UCID_From_UCIndex</b>.
	 * 	@param pc The ProductCluster relative to which the unique UserCluster ID should be retrieved.
	 * 	@param ucIdx The index of the UserCluster to return
	 * 	@return int 
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C00DA
	 */
	public int getUCID(ProductCluster pc, int ucIdx) throws AlkExcept 
	{
		final String mName = "getUCID";
		int o_ucid;
		Connection con = null;
        try {
			
			con = AlkConn.getConnection();
            CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_CLUSTER.sp_SEL_UCID_From_UCIndex(?,?,?,?)}");
            cs.setInt(1, pc.id);
            cs.setInt(2, ucIdx);
            cs.registerOutParameter(3, Types.INTEGER);
            cs.registerOutParameter(4, Types.INTEGER);
            cs.execute();

            o_ucid = cs.getInt(3);
            cs.close();
		} 

		catch (SQLException se) {
            throw new AlkExcept("SQL Exception in ClusterMgrDBUtils.getUCID: " + se.getMessage(), 6004);
        }
		finally {
			try { con.close(); } catch (Exception e) {}
		}
        return o_ucid;
	}
	
	/**
	 * Constructs a VMeansList from a ResultSet. The ResultSet <i>must</i> include the following data for each row:<br>
	 * <ul>
	 * <li>UserCluster ID</li>
	 * <li>Product ID</li>
	 * <li>average evaluation</li>
	 * </ul>
	 * In addition, the ResultSet <i>must</i> use the column names given in the class Alkindi.Services.Util.AlkConn.
	 * 
	 * @param rs The ResultSet to parse.
	 * @return VMeansList
	 * @throws SQLException
	 * @roseuid 3A7EF7E101F4
	 */
	private VMeansList parseVMLFromRS(final java.sql.ResultSet rs) throws SQLException 
	{
		//Initialize a return value
		//
		VMeansList vml = new VMeansList();
		//	Iterate over the rows in the recordset, putting the eval into the VMeans 
		//	object corresponding to the the UC.
		//
		while (rs.next()) {
			//	Retrieve the UC Idx for the Product-UC stat record
			//
			int ucID = rs.getInt(AlkConn.COL_UCIDX);
			
			//	Get the VMeans for the UC idx. If it doesn't exist, keep adding VMeans until it does.
			//
			VMeans vm = null;
			while (vm == null) {
				try {
					vm = vml.get(ucID);
				}
				catch(IndexOutOfBoundsException ie) {
					vml.add(new VMeans());
				}
			}
			//	Get the product ID and mean.
			//
			int prodID = rs.getInt(AlkConn.COL_PID);
			float eval = rs.getFloat(AlkConn.COL_AVGEVAL);
			//	Add the product mean to the VMeans
			//
			vm.putEval(prodID, eval);
		}
		return vml;
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_ALKINDEX_STAT</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660D001F
	 */
	public void recalcAlkindexStat() throws AlkExcept 
	{
		final String mName = "recalcAlkindexStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall("{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_ALKINDEX_STAT(?,?)}");

			cs.setInt(1, propAlkindexTopNProds);
			cs.registerOutParameter(2, Types.INTEGER);
			cs.execute();
			cs.close();
			instr.stopTiming();

       	} 
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2119);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_PC_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C0242
	 */
	public void recalcPCStat(int i_omega, int i_dscore) throws AlkExcept 
	{
		final String mName = "recalcPCStat";
		Connection con = null;

		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
			"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_PC_Stat(?,?,?)}");

			cs.setInt(1, i_omega);
			cs.setInt(2, i_dscore);
			cs.registerOutParameter(3, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(3);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in sp_INS_ReCalc_PC_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2108);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_PROD_PC_Stat</b>.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B660C03B9
	 */
	public void recalcProdPCStat() throws AlkExcept 
	{
		final String mName = "recalcProdPCStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_PROD_PC_Stat(?)}");

			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in sp_INS_ReCalc_PROD_PC_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2119);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_UPD_ReCalc_ProductScore</b>.
	 * 	<p><b><i>Important:</i></b>&nbsp;To consider all ProductClusters in the calculations, pass this method a ProductCluster object with an ID of zero.
	 * 	@param pc The ProductCluster to consider in calculations.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C035B
	 */
	public void recalcProdScore(ProductCluster pc) throws AlkExcept 
	{
		final String mName = "recalcProdScore";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs;
			int outParamIdx = 1;
			if (0 == pc.id){
				cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_UPD_ReCalc_ProductScore(?)}");
			} else {
				cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_UPD_ReCalc_ProductScore(?,?)}");
				cs.setLong(1, (long)pc.id);
				outParamIdx = 2;
			}
			cs.registerOutParameter(outParamIdx, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(outParamIdx);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in pkg_ALKINDI_STAT.sp_UPD_ReCalc_ProductScore", err);
			}
			instr.stopTiming();
		} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2117);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_Prod_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C0203
	 */
	public void recalcProdStat() throws AlkExcept 
	{
		final String mName = "recalcProdStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_Prod_Stat(?)}");

			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("General error in sp_INS_ReCalc_Prod_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2106);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_Prod_UC_Stat</b>.
	 * 	<p><i>Important:</i>&nbsp;To consider all ProductClusters in the calculations, pass this method a ProductCluster object with an ID of zero.
	 * 	@param pc The ProductCluster to consider in calculations.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C02AF
	 */
	public void recalcProdUCStat(ProductCluster pc) throws AlkExcept 
	{
		final String mName = "recalcProdUCStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs;
			int outParamIdx = 1;	
			if (0 == pc.id) {
				cs  = con.prepareCall(
					"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_Prod_UC_Stat(?)}");

			} else {
				cs  = con.prepareCall(
					"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_Prod_UC_Stat(?,?)}");
				cs.setLong(1, (long)pc.id);
				outParamIdx = 2;
			} 
			cs.registerOutParameter(outParamIdx, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(outParamIdx);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in pkg_ALKINDI_STAT.sp_INS_ReCalc_Prod_UC_Stat", err);
			}
			instr.stopTiming();
		} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2113);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_UC_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C0290
	 */
	public void recalcUCStat() throws AlkExcept 
	{
		final String mName = "recalcUCStat";
		Connection con = null;

		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_UC_Stat(?)}");

			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in sp_INS_ReCalc_UC_Stat", err);
			}
			instr.stopTiming();

       	}
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2109);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_USER_DATA_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C0399
	 */
	public void recalcUserDataStat() throws AlkExcept 
	{
		final String mName = "recalcUserDataStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_USER_DATA_Stat(?)}");

			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in sp_INS_ReCalc_USER_DATA_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2118);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_User_PC_Stat</b>.
	 * 	@param omega A parameter passed to the stored procedure which governs which products it considers for calculations.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C02DE
	 */
	public void recalcUserPCStat(int omega) throws AlkExcept 
	{
		final String mName = "recalcUserPCStat";
		Connection con = null;

		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_User_PC_Stat(?,?)}");

			cs.setInt(1, omega);
			cs.registerOutParameter(2, Types.INTEGER);
			cs.execute();
			int err =  cs.getInt(2);
			cs.close();

			if (err != 0) {
				throw new AlkExcept("Error in pkg_ALKINDI_STAT.sp_INS_ReCalc_User_PC_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2114);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_User_SF_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C033C
	 */
	public void recalcUserSFStat() throws AlkExcept 
	{
		final String mName = "recalcUserSFStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_User_SF_Stat(?)}");

			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0 ) {
				throw new AlkExcept("Error in pkg_ALKINDI_STAT.sp_INS_ReCalc_User_SF_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2116);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_User_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C0222
	 */
	public void recalcUserStat() throws AlkExcept 
	{
		final String mName = "recalcUserStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_User_Stat(?)}");

			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in sp_INS_ReCalc_User_Stat", err);
			}
			instr.stopTiming();
		} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2107);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_INS_ReCalc_User_UC_Stat</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C031C
	 */
	public void recalcUserUCStat() throws AlkExcept 
	{
		final String mName = "recalcUserUCStat";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall("{ call pkg_ALKINDI_STAT.sp_INS_ReCalc_User_UC_Stat(?)}");
			cs.registerOutParameter(1, Types.INTEGER);
			cs.execute();
			int err = cs.getInt(1);
			cs.close();
			if (err != 0) {
				throw new AlkExcept("Error in pkg_ALKINDI_STAT.sp_INS_ReCalc_User_UC_Stat", err);
			}
			instr.stopTiming();
       	} 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2115);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_STAT.sp_UPD_USER_SUBGROUP</b>.
	 * 	@return void
	 * 	@throws AlkExcept
	 * @roseuid 3A4B660C03D8
	 */
	public void updateUserSubGroups() throws AlkExcept 
	{
		final String mName = "updateUserSubGroups";
		Connection con = null;
		try {
			Instrumenter instr = Instrumenter.getInstance(logger, cName, mName);
			instr.startTiming();
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall(
				"{ call pkg_ALKINDI_STAT.sp_UPD_USER_SUBGROUP()}");

			cs.execute();
			cs.close();
			instr.stopTiming();
       	} 
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in " + mName + ": " + se.getMessage(), 6004);
   	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2119, e);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedures <b>pkg_ALKINDI_Cluster.sp_DEL_UC_By_PC</b> and <b>pkg_ALKINDI_Cluster.sp_INS_UC_By_PC_UC_INDEX</b>.
	 * 
	 * 	@param numUC The number of UserCluster to create in the database.
	 * 	@param pc The ProductCluster in question.
	 * @roseuid 3A4B660C00AB
	 */
	public void writeInitialUCForPC(int numUC, ProductCluster pc) throws AlkExcept 
	{
		final String mName = "writeInitialUCForPC";
		Connection con = null;	
		try {
			
			con = AlkConn.getConnection();

			// Delete all User Clusters from the Product Cluster
			//
            CallableStatement cs = con.prepareCall("{call pkg_ALKINDI_Cluster.sp_DEL_UC_By_PC(?)}");
			cs.setInt(1,pc.id);
			cs.execute();
			cs.close();

            CallableStatement cs1 = con.prepareCall("{call pkg_ALKINDI_Cluster.sp_INS_UC_By_PC_UC_INDEX(?,?)}");
			cs1.setInt(1,pc.id);

            for (int i = 0; i < numUC; i++) {
				cs1.setInt(2, i);
				cs1.execute();
            }
            cs1.close();
            
        }
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in : " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2111);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.sp_INS_USER_Into_UC</b>.
	 * 	@param userID The ID of the SystemUser in question.
	 * 	@param ucIdx The index of the UserCluster to write.
	 * 	@param pc The ID of the ProductCluster relative to which the UserCluster exists.
	 * @roseuid 3A4B660C0138
	 */
	public void writeSingleUCMembership(SystemUser user, int ucIdx, ProductCluster pc) throws AlkExcept 
	{
		final String mName = "writeSingleUCMembership";
		Connection con = null;

		try {
			
			con = AlkConn.getConnection();
			CallableStatement cs  = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.sp_INS_USER_Into_UC (?,?,?,?)}");
			cs.setLong(1,user.id);
			cs.setInt(2,ucIdx);
			cs.setInt(3,pc.id);
			cs.registerOutParameter(4, Types.INTEGER);
			cs.execute();
			cs.close();
        } 
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in : " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2112);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * 	Stores the membership for the given UserClusters relative to the given ProductCluster.
	 * 	Calls the Stored Procedure <b>pkg_ALKINDI_CLUSTER.sp_INS_USER_Into_UC_With_UCID</b>.
	 * 	<p><b>Note:</b>&nbsp;This routine does not begin or otherwise manipulate transactions.
	 * 	@param ucl The UserClusterList to store in the database.
	 * 	@param pc The ProductCluster in question.
	 * @roseuid 3A4B660C007D
	 */
	public void writeUCMembershipForPC(UserClusterList ucl, ProductCluster pc) throws AlkExcept 
	{
		final String mName = "writeUCMembershipForPC";
		Connection con = null;

        try {
			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall("{ call pkg_ALKINDI_CLUSTER.sp_INS_USER_Into_UC_With_UCID(?,?,?,?)}");
			Iterator clusterIt = ucl.iterator();
			int clusterIdx = 0;
			//	Traverse the list of user clusters.
			//
			while(clusterIt.hasNext()) {
				UserCluster uc = (UserCluster)clusterIt.next();
				//	Obtain the correct ID for the user cluster
				//
				int ucid = getUCID(pc, clusterIdx);
				//	Traverse the UserRatings in the UserCluster in order
				//	to effectively traverse the users.
				//
				Iterator userRatingsIt = uc.userRatingsList.userRatingsIterator();
				while (userRatingsIt.hasNext()) {
					//	Write the user to the database as a cluster member.
					//
					UserRatings ur = (UserRatings)userRatingsIt.next();
					cs.setLong(1, ur.user().id);
					cs.setInt(2, ucid);
					cs.setInt(3,pc.id);
					cs.registerOutParameter(4, Types.INTEGER);	
					cs.executeUpdate();
					int err = cs.getInt(4);
					if (err != 0)
						throw new AlkExcept("Error in sp_INS_USER_Into_UC_With_UCID", err);
                }
				clusterIdx++;
            }

            cs.close();

        } 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in : " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2111);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
	
	/**
	 * Writes the given VMeans for the given ProductCluster.
	 * <p>
	 * <b>Note:</b>&nbsp; This routine does not manage transactions.
	 * @param vml The VMeans list to store in the database.
	 * @param pc The ProductCluster in question.
	 * @return void
	 * @throws AlkExcept
	 * @roseuid 3A4B660C004E
	 */
	public void writeVMeansForPC(final VMeansList vml, ProductCluster pc) throws AlkExcept 
	{
		final String mName = "writeVMeansForPC";
		Connection con = null;

        try {
   			
			con = AlkConn.getConnection();
			CallableStatement cs = con.prepareCall( "{call pkg_ALKINDI_CLUSTER.updateAvgProdRatingByUC(?,?,?,?)}");
			
			//	Iterate over VMeans [clusters] in list.
			//
			for (int vmIdx = vml.size(); --vmIdx > -1; ) {
				VMeans vm = vml.get(vmIdx);
				int ucID = getUCID(pc, vmIdx);	
				//	For each VMeans, iterate over the products and 
				//	call the SP for each product/mean pair.
				//
				int[] prods = vm.getProductArray();
				int numProds = prods.length;
				float[] evals = vm.getEvalsArray();
				for (int idx = 0; idx < numProds; idx ++ ) {
					cs.setInt(1, prods[idx]);
					cs.setInt(2, ucID);
					cs.setFloat(3, evals[idx]);
					cs.registerOutParameter(4, Types.INTEGER);
					cs.execute();
				}
			}
            cs.close();
        } 
		catch (AlkExcept ae) {
			throw ae;
		}
		catch (SQLException se) {
			throw new AlkExcept("SQL Exception in : " + mName + ": " + se.getMessage(), 6004);
       	}
		catch(Exception e) {
			throw new AlkExcept("General error in " + mName, 2110);
		}
		finally {
			try { con.close(); } catch(Exception e) {}
		}
	}
}
